//
//  ViewModel.swift
//  BreastFeedingDemo
//
//  Created by Jay Zhang on 2022/6/5.
//

import Flutter
import FlutterPluginRegistrant
import Foundation

class ViewModel {
    enum Posture: Int {
        case cradle = 0
        case lying = 1
    }

    /// 语言枚举，enum代表语言包文件名，rawValue代表语言选项
    enum Language: String, CaseIterable {
        case english = "English"
        case french = "Français"
        case spanish = "Español"
    }

    var env: String = "testing"
    var apiKey: String = "DLAB-customername-12345687"
    var region: String = "apac"

    var language: Language = .english {
        didSet {
            setupText()
        }
    }

    var flutterEngine: FlutterEngine!
    var posture: Posture = .cradle

    var string: StringModel!
    var isFirstLoad: Bool = true

    var languagePack: [String: Any] {
        let languageFile = "\(language)".appending(".json")
        let path: String = Bundle.main.path(forResource: languageFile, ofType: nil)!
        let nsUrl = URL(fileURLWithPath: path)
        let nsData = try! Data(contentsOf: nsUrl)
        let json = try! JSONSerialization.jsonObject(with: nsData) as! [String: Any]
        return json
    }

    var flutterArguments: String {
        let arguments: [String: Any] = [
            "lying": posture.rawValue,
            "languagePack": languagePack,
            "env": env,
            "apiKey": apiKey,
            "region": region,
        ]

        let data = try? JSONSerialization.data(withJSONObject: arguments, options: [])
        let str = String(data: data!, encoding: String.Encoding.utf8)
        return str!
    }

    var methodChannel: FlutterMethodChannel!
    var completeHandler: (() -> Void)?
    var teachingEndHandler: (() -> Void)?

    init() {
        language = .english
        setupText()
        runFlutterEngine()
        methodChannel = setupMethodCallHandler()
    }

    func runFlutterEngine() {
        flutterEngine = FlutterEngine(name: "my flutter engine")
        flutterEngine.run()
        GeneratedPluginRegistrant.register(with: flutterEngine)
    }

    func setupMethodCallHandler() -> FlutterMethodChannel {
        let binaryMessenger = flutterEngine.binaryMessenger
        let methodChannel = FlutterMethodChannel(name: "com.wmdigit.breastcoachai.native", binaryMessenger: binaryMessenger)
        methodChannel.setMethodCallHandler { [weak self]
            (call: FlutterMethodCall, result: @escaping FlutterResult) in
                if let self = self {
                    switch call.method {
                    case "init":
                        let params = self.initLanguagePack() // 生成JSON格式的语言包
                        result(params) // 初始化
                    case "teachingEnd":
                        self.methodChannel.invokeMethod("dispose", arguments: nil)
                        // ["totalDuration": 3, "reasonText": Concern about my personal privacy, "status": incomplete, "incompleteReasonCode": 2]
                        let argument: [String: Any] = try! JSONSerialization.jsonObject(with: (call.arguments as! String).data(using: .utf8)!) as! [String: Any]

                        let status: String = argument["status"] as! String
                        if status == "complete" {
                            self.completeHandler?()
                            return
                        }
                        self.teachingEndHandler?()
                    default:
                        // Unrecognized method name
                        print("Unrecognized method name: \(call.method)")
                    }
                }
        }
        return methodChannel
    }

    fileprivate func initLanguagePack() -> String {
        return viewModel.flutterArguments
    }

    fileprivate func setupText() {
        let languageFile = "string_\(language)".appending(".json")
        let path: String = Bundle.main.path(forResource: languageFile, ofType: nil)!
        let nsUrl = URL(fileURLWithPath: path)
        let nsData = try! Data(contentsOf: nsUrl)
        let json = try! JSONSerialization.jsonObject(with: nsData) as! [String: Any]

        string = .init(fromDictionary: json)
    }
}
