//
//  ICRHTTPController.m
//  Cruiser
//
//  Created by Xummer on 3/26/15.
//  Copyright (c) 2015 Xummer. All rights reserved.
//

#import "ICRHTTPController.h"
#import "ICRUserUtil.h"
#import "AFNetworking.h"
#import "AFNetworkActivityIndicatorManager.h"

#define MAX_CONCURRENCY_UPLOAD      1
#define MAX_CONCURRENCY_DOWNLOAD    3

#define ERROR_PARAMETER             @"Error Parameter(s)"

NSString * const ICRHTTPErrorDomain = @"ICRHTTPErrorDomain";

typedef NS_ENUM(NSUInteger, ICRHTTPAction) {
    kICRHTTP_Dummy = 0,
    // User
    kICRHTTP_UserLogin,
    kICRHTTP_UserResetPassword,
    
    // Data
    kICRHTTP_StoreQuery,
    kICRHTTP_CurrentOrg,
    kICRHTTP_PostQuery,
    
    // Board
    kICRHTTP_BoardQuery,
    kICRHTTP_FetchBoard,
    kICRHTTP_ReadBoard,
    
    // Patrol
    kICRHTTP_PatrolQuery,
    kICRHTTP_AnswerStoreResult,
    kICRHTTP_PatrolAnswer,
    kICRHTTP_PatrolOneAnswer,
    
    // Task
    kICRHTTP_TaskQuery,
    kICRHTTP_TaskNew,
    kICRHTTP_TaskResult,
    
    // Signup
    kICRHTTP_Signup,
    
    // Attachment
    kICRHTTP_AttachmentAdd,
    kICRHTTP_AttachmentList,
    kICRHTTP_AttachmentDownload,
    kICRHTTP_AttachmentAddDirect,
    kICRHTTP_AttachmentDownloadDirect,
    
    // Version
    kICRHTTP_VersionFetch,
    
    //
};

static NSString * const ICRHTTPInterface[] = {
    [ kICRHTTP_Dummy ]                      = @"",
    
    // User
    [ kICRHTTP_UserLogin ]                  = @"user/login",
    [ kICRHTTP_UserResetPassword ]          = @"user/resetPassword",
    
    // Data
    [ kICRHTTP_StoreQuery ]                 = @"data/store/query",
    [ kICRHTTP_CurrentOrg ]                 = @"data/organization/currentOrg",
    [ kICRHTTP_PostQuery ]                  = @"data/post/query",
    
    // Board
    [ kICRHTTP_BoardQuery ]                 = @"board/query",
    [ kICRHTTP_FetchBoard ]                 = @"board",
    [ kICRHTTP_ReadBoard ]                  = @"board/read",
    
    // Patrol
    [ kICRHTTP_PatrolQuery ]                = @"patrol/query",
    [ kICRHTTP_AnswerStoreResult ]          = @"patrol/answer/storeResult",
    [ kICRHTTP_PatrolAnswer ]               = @"patrol/answer",
    [ kICRHTTP_PatrolOneAnswer ]            = @"patrol/oneAnswer",
    
    // Task
    [ kICRHTTP_TaskQuery ]                  = @"task/query",
    [ kICRHTTP_TaskNew ]                    = @"task/new",
    [ kICRHTTP_TaskResult ]                 = @"task/result",
    
    // Signup
    [ kICRHTTP_Signup ]                     = @"signup/signup",
    
    // Attachment
    [ kICRHTTP_AttachmentAdd ]              = @"attachment/add",
    [ kICRHTTP_AttachmentList ]             = @"attachment/list",
    [ kICRHTTP_AttachmentDownload ]         = @"attachment/download",
    [ kICRHTTP_AttachmentAddDirect ]        = @"attachment/addDirect",
    [ kICRHTTP_AttachmentDownloadDirect ]   = @"attachment/downloadDirect",
    
    // Version
    [ kICRHTTP_VersionFetch ]               = @"version/fetch",
};

@interface ICRHTTPController ()
/**
 *  The upload operation queue on which upload request operations are scheduled and run.
 */
@property (nonatomic, strong) NSOperationQueue *uploadOperationQueue;

/**
 *  The download operation queue on which download request operations are scheduled and run.
 */
@property (nonatomic, strong) NSOperationQueue *downloadOperationQueue;


@end

@implementation ICRHTTPController

#pragma mark - Class Method

+ (instancetype)sharedController {
    static ICRHTTPController *_sharedController = nil;
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        _sharedController = [[self alloc] init];
    });
    
    return _sharedController;
}

+ (NSString *)UrlForPluginHTTPAction:(ICRHTTPAction)action {
    return [HTTP_REST_API_BASE_URL stringByAppendingFormat:@"/%@", ICRHTTPInterface[ action ]];
}

+ (NSError *)ErrorWithMsg:(NSString *)nsErrorMsg code:(NSInteger)uiCode {
    NSDictionary *userInfo =
    nsErrorMsg ? @{ NSLocalizedFailureReasonErrorKey : nsErrorMsg } : nil;
    
    NSError *error =
    [[NSError alloc] initWithDomain:ICRHTTPErrorDomain
                               code:uiCode userInfo:userInfo];
    
    return error;
}

+ (id)GetErrorMsgFromOperation:(AFHTTPRequestOperation *)operation error:(NSError *)error
{
    id data = operation.responseObject;
    if ([data isKindOfClass:[NSDictionary class]]) {
        NSString *errorMsg = data[ @"message" ];
        return [[self class] ErrorWithMsg:errorMsg code:[[operation response] statusCode]];
    }
    else {
        return error;
    }
}


#pragma mark - Life Cycle
- (id)init {
    self = [super init];
    if (!self) {
        return nil;
    }
    
    [[AFNetworkActivityIndicatorManager sharedManager] setEnabled:YES];
    
    self.uploadOperationQueue = [[NSOperationQueue alloc] init];
    [self.uploadOperationQueue setMaxConcurrentOperationCount:MAX_CONCURRENCY_UPLOAD];
    
    self.downloadOperationQueue = [[NSOperationQueue alloc] init];
    [self.downloadOperationQueue setMaxConcurrentOperationCount:MAX_CONCURRENCY_DOWNLOAD];
    
    return self;
}

#pragma mark - Private Method
- (AFJSONRequestSerializer *)requestNeedToken:(BOOL)bIsNeedToken
                               acceptTypeJson:(BOOL)bAcceptJson
                                      failure:(void (^)(AFHTTPRequestOperation *operation, NSError *error))fail
{
    AFJSONRequestSerializer *requestSerializer = [AFJSONRequestSerializer serializer];
    if (bIsNeedToken) {
        
        ICRUserUtil *user = [ICRUserUtil sharedInstance];
        
        if ([user.token length] > 0) {
            if (!requestSerializer.HTTPRequestHeaders[ @"token" ]) {
                [requestSerializer setValue:@"application/json" forHTTPHeaderField:@"token"];
            }
        }
        else {
            NSDictionary *userInfo = @{ NSLocalizedFailureReasonErrorKey : @"The authorization must not be nil." };
            NSError *error = [[NSError alloc] initWithDomain:ICRHTTPErrorDomain code:0 userInfo:userInfo];
            fail(nil, error);
        }
    }
    
    if (bAcceptJson) {
        if (!requestSerializer.HTTPRequestHeaders[ @"Accept" ]) {
            [requestSerializer setValue:@"application/json" forHTTPHeaderField:@"Accept"];
        }
    }
    
    return requestSerializer;
}

- (void)GET:(NSString *)urlStr
 parameters:(id)parameters
  needToken:(BOOL)bIsNeedToken
acceptTypeJson:(BOOL)bAcceptJson
    success:(void (^)(AFHTTPRequestOperation *operation, id responseObject))succ
    failure:(void (^)(AFHTTPRequestOperation *operation, NSError *error))fail
{
    AFJSONRequestSerializer *requestSerializer =
    [self requestNeedToken:bIsNeedToken acceptTypeJson:bAcceptJson failure:fail];
    
    void (^success)(AFHTTPRequestOperation *operation, id responseObject) = ^(AFHTTPRequestOperation *operation, id responseObject) {
        succ(operation, responseObject);
    };
    
    void (^failure)(AFHTTPRequestOperation *operation, NSError *error) = ^(AFHTTPRequestOperation *operation, NSError *error) {
        fail(operation, [[self class] GetErrorMsgFromOperation:operation error:error]);
    };
    
    AFHTTPRequestOperationManager *manager = [AFHTTPRequestOperationManager manager];
    manager.requestSerializer = requestSerializer;
    
    // The request add in operation
    [manager GET:urlStr
      parameters:parameters
         success:success
         failure:failure];
}

- (void)POST:(NSString *)urlStr
  parameters:(id)parameters
   needToken:(BOOL)bIsNeedToken
acceptTypeJson:(BOOL)bAcceptJson
     success:(void (^)(AFHTTPRequestOperation *operation, id responseObject))succ
     failure:(void (^)(AFHTTPRequestOperation *operation, NSError *error))fail
{
    AFJSONRequestSerializer *requestSerializer =
    [self requestNeedToken:bIsNeedToken acceptTypeJson:bAcceptJson failure:fail];
    
    void (^success)(AFHTTPRequestOperation *operation, id responseObject) = ^(AFHTTPRequestOperation *operation, id responseObject) {
        succ(operation, responseObject);
    };
    
    void (^failure)(AFHTTPRequestOperation *operation, NSError *error) = ^(AFHTTPRequestOperation *operation, NSError *error) {
        fail(operation, [[self class] GetErrorMsgFromOperation:operation error:error]);
    };
    
    AFHTTPRequestOperationManager *manager = [AFHTTPRequestOperationManager manager];
    manager.requestSerializer = requestSerializer;
    
    // The request add in operation
    [manager POST:urlStr
       parameters:parameters
          success:success
          failure:failure];
}

- (void)PATCH:(NSString *)urlStr
   parameters:(id)parameters
    needToken:(BOOL)bIsNeedToken
acceptTypeJson:(BOOL)bAcceptJson
      success:(void (^)(AFHTTPRequestOperation *operation, id responseObject))succ
      failure:(void (^)(AFHTTPRequestOperation *operation, NSError *error))fail
{
    AFJSONRequestSerializer *requestSerializer =
    [self requestNeedToken:bIsNeedToken acceptTypeJson:bAcceptJson failure:fail];
    
    void (^success)(AFHTTPRequestOperation *operation, id responseObject) = ^(AFHTTPRequestOperation *operation, id responseObject) {
        succ(operation, responseObject);
    };
    
    void (^failure)(AFHTTPRequestOperation *operation, NSError *error) = ^(AFHTTPRequestOperation *operation, NSError *error) {
        fail(operation, [[self class] GetErrorMsgFromOperation:operation error:error]);
    };
    
    AFHTTPRequestOperationManager *manager = [AFHTTPRequestOperationManager manager];
    manager.requestSerializer = requestSerializer;
    
    // The request add in operation
    [manager PATCH:urlStr
        parameters:parameters
           success:success
           failure:failure];
}

- (void)DELETE:(NSString *)urlStr
    parameters:(id)parameters
     needToken:(BOOL)bIsNeedToken
acceptTypeJson:(BOOL)bAcceptJson
       success:(void (^)(AFHTTPRequestOperation *operation, id responseObject))succ
       failure:(void (^)(AFHTTPRequestOperation *operation, NSError *error))fail
{
    AFJSONRequestSerializer *requestSerializer =
    [self requestNeedToken:bIsNeedToken acceptTypeJson:bAcceptJson failure:fail];
    
    void (^success)(AFHTTPRequestOperation *operation, id responseObject) = ^(AFHTTPRequestOperation *operation, id responseObject) {
        succ(operation, responseObject);
    };
    
    void (^failure)(AFHTTPRequestOperation *operation, NSError *error) = ^(AFHTTPRequestOperation *operation, NSError *error) {
        fail(operation, [[self class] GetErrorMsgFromOperation:operation error:error]);
    };
    
    AFHTTPRequestOperationManager *manager = [AFHTTPRequestOperationManager manager];
    manager.requestSerializer = requestSerializer;
    
    // The request add in operation
    [manager DELETE:urlStr
         parameters:parameters
            success:success
            failure:failure];
}

#pragma mark - User
- (void)doLoginWithUserName:(NSString *)nsUserName
                   password:(NSString *)nsPassword
               registerCode:(NSString *)nsRegisterCode
                    success:(void (^)(id data))succ
                    failure:(void (^)(id data))fail
{
    if (!nsUserName || !nsPassword || !nsRegisterCode) {
        fail( [[self class] ErrorWithMsg:ERROR_PARAMETER code:0] );
        return;
    }
    
    void (^success)(AFHTTPRequestOperation *operation, id responseObject) = ^(AFHTTPRequestOperation *operation, id responseObject) {
        CLog(@"%@",responseObject);
        
        id dictResult = responseObject[ @"result" ];
        if (dictResult) {
            
            ICRUserUtil *userUtil = [ICRUserUtil sharedInstance];
            userUtil.userName = nsUserName;
            userUtil.password = nsPassword;
            userUtil.registerCode = nsRegisterCode;
            userUtil.token = dictResult[ @"token" ];
            userUtil.displayName = dictResult[ @"name" ];
            userUtil.orgCode = dictResult[ @"orgCode" ];
            userUtil.orgName = dictResult[ @"orgName" ];
            
            [userUtil saveArchive];
            
            if (succ) {
                succ( dictResult );
            }
        }
        else {
            if (fail) {
                fail( nil );
            }
        }
    };
    
    void (^failure)(AFHTTPRequestOperation *operation, NSError *error) = ^(AFHTTPRequestOperation *operation, NSError *error) {
        CLog(@"%@",error);
        if (fail) {
            fail( error );
        }
    };
    
    NSDictionary *dictParametes =
    @{ @"password" : nsPassword,
       @"regesterCode" : nsRegisterCode,
      };
    
    NSString *urlStr = [[[self class] UrlForPluginHTTPAction:kICRHTTP_UserLogin] stringByAppendingFormat:@"/%@", nsUserName];
    [self POST:urlStr
    parameters:dictParametes
     needToken:NO
acceptTypeJson:YES
       success:success
       failure:failure];
}

- (void)doChangePassword:(NSString *)nsPassword
             newPassword:(NSString *)nsNewPassword
                 success:(void (^)(id data))succ
                 failure:(void (^)(id data))fail
{
    if (!nsPassword || !nsNewPassword) {
        fail( [[self class] ErrorWithMsg:ERROR_PARAMETER code:0] );
        return;
    }
    
    void (^success)(AFHTTPRequestOperation *operation, id responseObject) = ^(AFHTTPRequestOperation *operation, id responseObject) {
        
    };
    
    void (^failure)(AFHTTPRequestOperation *operation, NSError *error) = ^(AFHTTPRequestOperation *operation, NSError *error) {
        
    };
    
    NSDictionary *dictParametes =
    @{ @"password" : nsPassword,
       @"newPassword" : nsNewPassword,
       };
    
    NSString *urlStr = [[self class] UrlForPluginHTTPAction:kICRHTTP_UserResetPassword];
    [self POST:urlStr
    parameters:dictParametes
     needToken:NO
acceptTypeJson:YES
       success:success
       failure:failure];
}


#pragma mark - Data
- (void)doGetStoreListFromUpdateTime:(NSTimeInterval)updateTime
                            position:(NSUInteger)uiPosition
                                size:(NSUInteger)uiSize
                             success:(void (^)(id data))succ
                             failure:(void (^)(id data))fail
{

}



@end
