//
//  ICRTaskListViewController.m
//  XFFruit
//
//  Created by Xummer on 4/9/15.
//  Copyright (c) 2015 Xummer. All rights reserved.
//

#import "ICRTaskListViewController.h"
#import "ICRTaskDetailViewController.h"
#import "ICRTaskResultViewController.h"

#import "ICRTaskItemView.h"
#import "ICRTask.h"

typedef NS_ENUM(NSUInteger, TaskSegmentsType) {
    kTaskSegUnHandled = 0,
    kTaskSegHandled,
    kTaskSegMine,
};

static NSString *TaskCellIdentifier = @"TaskCell";

@interface ICRTaskListViewController ()
<
    UITableViewDataSource,
    UITableViewDelegate,
    IBTScrollViewRefreshDelegate
>

@property (strong, nonatomic) IBTRefreshTableView *m_tableView;
@property (strong, nonatomic) IBTSegmentContainer *m_segContainer;
@property (weak,   nonatomic) IBTSegmentedControl *m_segmentControl;

@property (strong, nonatomic) NSArray *m_arrAllData;
@property (strong, nonatomic) NSArray *m_arrData;

@end

@implementation ICRTaskListViewController

#pragma mark - Life Cycle
- (void)viewDidLoad {
    [super viewDidLoad];
    // Do any additional setup after loading the view
    self.title = [IBTCommon localizableString:@"Task List"];
    
    [self initSegmentConatinerFromOriginY:0];
    [self initTableView];
    
    [self fetchTaskList];
}

- (void)viewWillAppear:(BOOL)animated {
    [super viewWillAppear:animated];
    
    [self fetchDataMore:NO];
}

- (void)didReceiveMemoryWarning {
    [super didReceiveMemoryWarning];
    // Dispose of any resources that can be recreated.
    
}

#pragma mark - Setter
- (void)setM_arrAllData:(NSArray *)arrAllData {
    if ([_m_arrAllData isEqualToArray:arrAllData]) {
        return;
    }
    
    _m_arrAllData = arrAllData;
    
    [self updateDisplayDataWithType:_m_segmentControl.selectedSegmentIndex];
}

#pragma mark - Private Method
- (void)initTableView {
    CGFloat dy = CGRectGetMaxY(_m_segContainer.frame);
    CGRect frame = self.view.bounds;
    frame.origin.y = dy;
    frame.size.height -= CGRectGetMinY(frame);
    
    self.m_tableView =
    [[IBTRefreshTableView alloc] initWithFrame:frame style:UITableViewStylePlain];
    [_m_tableView registerClass:[IBTTableViewCell class]
       forCellReuseIdentifier:TaskCellIdentifier];
    [_m_tableView autoresizingWithStrechFullSize];
    _m_tableView.rowHeight = ICR_TASK_ITEM_HEIGTH;
    _m_tableView.dataSource = self;
    _m_tableView.delegate = self;
    _m_tableView.refreshDelegate = self;
    [_m_tableView addRefreshControlWithText:[IBTCommon localizableString:@""]];
    [_m_tableView addLoadMoreFootWithText:[IBTCommon localizableString:@"Load more"]];
    [self.view addSubview:_m_tableView];
}

- (void)initSegmentConatinerFromOriginY:(CGFloat)y {
    
    if (_m_segContainer) {
        return;
    }
    
    NSArray *arrSegments =
    @[ [IBTCommon localizableString:@"Pending"],
       [IBTCommon localizableString:@"Handled"],
       [IBTCommon localizableString:@"Post by Me"], ];
    self.m_segContainer = [[IBTSegmentContainer alloc] initWithItems:arrSegments];
    self.m_segmentControl = _m_segContainer.segmentControl;
    _m_segmentControl.selectedSegmentIndex = 0;
    _m_segContainer.frame = (CGRect){
        .origin.x = 0,
        .origin.y = y,
        .size.width = CGRectGetWidth(self.view.bounds),
        .size.height = IBT_SEGMENT_CONTAINER_DEFUALT_HEIGHT
    };
    [self.view addSubview:_m_segContainer];
    
    [_m_segmentControl addTarget:self action:@selector(onSegmentsTapped:)
                forControlEvents:UIControlEventValueChanged];
}

#pragma mark - Action
- (void)onSegmentsTapped:(id)sender {
    UISegmentedControl *segmentedCtrl = sender;
    [self updateDisplayDataWithType:segmentedCtrl.selectedSegmentIndex];
}

#pragma mark - UITableViewDataSource

- (NSInteger)numberOfSectionsInTableView:(UITableView *)tableView {
    return 1;
}

- (NSInteger)tableView:(UITableView *)tableView
 numberOfRowsInSection:(NSInteger)section
{
    return [_m_arrData count];
}

- (UITableViewCell *)tableView:(UITableView *)tableView
         cellForRowAtIndexPath:(NSIndexPath *)indexPath
{
    UITableViewCell *cell =
    [tableView dequeueReusableCellWithIdentifier:TaskCellIdentifier forIndexPath:indexPath];
    
    [self configureCell:cell forRowAtIndexPath:indexPath];
    
    return cell;
}

- (void)configureCell:(UITableViewCell *)cell
    forRowAtIndexPath:(NSIndexPath *)indexPath
{
    cell.accessoryType = UITableViewCellAccessoryDisclosureIndicator;
    
    id data = _m_arrData[ indexPath.row ];
    
    UIView *contentView = cell.contentView;
    
    ICRTaskItemView *itemView = [contentView viewWithClass:[ICRTaskItemView class]];
    if (!itemView) {
        itemView = [[ICRTaskItemView alloc] initWithFrame:contentView.bounds];
        [itemView autoresizingWithStrechFullSize];
        [contentView addSubview:itemView];
    }
    [itemView updateWithTask:data];
}

#pragma mark - UITableViewDelegate

- (void)tableView:(UITableView *)tableView
didSelectRowAtIndexPath:(NSIndexPath *)indexPath
{
    [tableView deselectRowAtIndexPath:indexPath animated:YES];
    
    ICRTask *data = _m_arrData[ indexPath.row ];
   
    if (data.processResult) {
        [self pushToTaskReault:data];
    }
    else if ([data.state isEqualToString:TaskState[ kICRTaskLocalCreated ]])
    {
        [self pushToTaskDetail:data];
    }
    else if ([data.state isEqualToString:TaskState[ kICRTaskLocalFinished ]])
    {
        [self pushToTaskReault:data];
    }
}

- (void)scrollViewDidScroll:(UIScrollView *)scrollView {
    if (scrollView == _m_tableView) {
        [_m_tableView tableViewDidScroll:scrollView];
    }
}

- (void)scrollViewDidEndDragging:(UIScrollView *)scrollView willDecelerate:(BOOL)decelerate {
    
    if (scrollView == _m_tableView) {
        [_m_tableView tableviewDidEndDragging:scrollView];
    }
}

#pragma mark - IBTScrollViewRefreshDelegate

- (void)startRefreshData:(UIScrollView *)scrollView {
    [self fetchDataMore:NO];
}

- (void)endRefreshData:(UIScrollView *)scrollView {
    
}

- (void)startLoadMoreData:(UIScrollView *)scrollView {
    [self fetchDataMore:YES];
}

- (void)endLoadMoreData:(UIScrollView *)scrollView {
    
}

#pragma mark - Action
- (void)pushToTaskDetail:(ICRTask *)task {
    ICRTaskDetailViewController *dVC =
    [[ICRTaskDetailViewController alloc] initWithTaskData:task];
    [self PushViewController:dVC animated:YES];
}

- (void)pushToTaskReault:(ICRTask *)task {
    ICRTaskResultViewController *rVC =
    [[ICRTaskResultViewController alloc] initWithTaskData:task];
    [self PushViewController:rVC animated:YES];
}


#pragma mark - Data Update

- (void)fetchTaskList {
    ICRDatabaseFetchBlock fetchBlk = ^FMResultSet *(FMDatabase *db) {
        NSString * sql = [NSString stringWithFormat:@"SELECT * FROM %@ ORDER BY %@ DESC", [ICRTask TableName], @"requireDate"];
        return [db executeQuery:sql];
    };
    
    __weak typeof(self)weakSelf = self;
    ICRDatabaseFetchResultsBlock fetchResultsBlk = ^(NSArray *fetchedObjects) {
        __strong __typeof(weakSelf)strongSelf = weakSelf;
        strongSelf.m_arrAllData = fetchedObjects;
    };
    
    ICRDataBaseController *dbCtrl = [ICRDataBaseController sharedController];
    [dbCtrl runFetchForClass:[ICRTask class]
                  fetchBlock:fetchBlk
           fetchResultsBlock:fetchResultsBlk];
}

- (void)updateDisplayDataWithType:(TaskSegmentsType)eType {
    
    NSString *filter = nil;
    NSArray *args = nil;
    
    switch (eType) {
        case kTaskSegUnHandled:
        {
            filter = @"(%K = %@)";
            args = @[ @"state", TaskState[kICRTaskLocalCreated]];
        }
            break;
        case kTaskSegHandled:
        {
            filter = @"(%K = %@)";
            args = @[ @"state", TaskState[kICRTaskLocalFinished]]
                      ;
        }
            break;
        case kTaskSegMine:
        {
            filter = @"(%K == %@)";
            args =@[ @"state", TaskState[kICRTaskLocalCreated]];//创建人是自己的任务
        }
            break;
            
        default:
            break;
    }
    if (filter) {
        NSPredicate *predicate = [NSPredicate predicateWithFormat:filter argumentArray:args];
        
        self.m_arrData = [_m_arrAllData filteredArrayUsingPredicate:predicate];
    }
    else {
        self.m_arrData = nil;
    }
    
    [self.m_tableView reloadData];
}

#pragma mark - Get Data

- (void)fetchDataMore:(BOOL)more {
    __weak typeof(self)weakSelf = self;
    void(^succ)(id) = ^(id data) {
        __strong __typeof(weakSelf)strongSelf = weakSelf;
        [strongSelf fetchTaskList];
        if (!more) {
            [_m_tableView endRefreshWithState:kRefreshStateFinished];
        }
        else {
            [_m_tableView endLoadMoreWithState:kLoadStateFinished];
        }

    };
    
    void(^fail)(id) = ^(id data) {
        [IBTLoadingView showTips:data];
        if (!more) {
            [_m_tableView endRefreshWithState:kRefreshStateFailed];
        }
        else {
            [_m_tableView endLoadMoreWithState:kLoadStateFailed];
        }

    };
    
    ICRHTTPController *httpCtrl = [ICRHTTPController sharedController];
    
    ICRTask *data = [_m_arrData lastObject];
    
    [httpCtrl doGetTaskListFromUpdateTime:more ? data.processDate : nil position:0 size:100
                                  success:succ failure:fail];
    
}

@end
