import { ref } from 'vue'
import {
  ElMessage,
} from 'element-plus'

export function useCropBox(form: any, canvas: any, ctx: any) {
  const isDrawing = ref(false)
  const startX = ref(0)
  const startY = ref(0)
  const endX = ref(0)
  const endY = ref(0)
  const boxes = ref<Array<{ start: number[], end: number[] }>>([])

  const draggingBoxIndex = ref<number | null>(null) // 正在拖动的框索引
  const resizingBoxIndex = ref<number | null>(null) // 正在缩放的框索引
  const resizeHandle = ref<'nw' | 'ne' | 'sw' | 'se' | null>(null) // 缩放角方向
  const dragStart = ref({ x: 0, y: 0 }) // 拖动起始点
  const boxStart = ref({ x: 0, y: 0, width: 0, height: 0 }) // 框原始位置

  function isInsideBox(x: number, y: number, box: any): boolean {
    return (
      x >= box.start[0] &&
      x <= box.end[0] &&
      y >= box.start[1] &&
      y <= box.end[1]
    )
  }
  function getHandleAt(x: number, y: number, box: any): 'nw' | 'ne' | 'sw' | 'se' | null {
    const size = 10
    const [x1, y1] = box.start
    const [x2, y2] = box.end

    if (Math.abs(x - x1) < size && Math.abs(y - y1) < size) return 'nw'
    if (Math.abs(x - x2) < size && Math.abs(y - y1) < size) return 'ne'
    if (Math.abs(x - x1) < size && Math.abs(y - y2) < size) return 'sw'
    if (Math.abs(x - x2) < size && Math.abs(y - y2) < size) return 'se'
    return null
  }

  function onMouseDown(e: MouseEvent) {
    const rect = canvas.value!.getBoundingClientRect()
    const x = e.clientX - rect.left
    const y = e.clientY - rect.top

    // 检查是否点击到了某个框的角
    for (let i = 0; i < boxes.value.length; i++) {
      const box = boxes.value[i]
      const handle = getHandleAt(x, y, box)
      if (handle) {
        resizingBoxIndex.value = i
        resizeHandle.value = handle
        return
      }

      if (isInsideBox(x, y, box)) {
        draggingBoxIndex.value = i
        dragStart.value = { x, y }
        const [bx1, by1] = box.start
        const [bx2, by2] = box.end
        boxStart.value = {
          x: bx1,
          y: by1,
          width: bx2 - bx1,
          height: by2 - by1
        }
        return
      }
    }
  }
  function onMouseMove(e: MouseEvent) {
    const rect = canvas.value!.getBoundingClientRect()
    const x = e.clientX - rect.left
    const y = e.clientY - rect.top

    if (resizingBoxIndex.value !== null && resizeHandle.value) {
      const i = resizingBoxIndex.value
      const box = boxes.value[i]
      const [bx1, by1] = box.start
      const [bx2, by2] = box.end

      switch (resizeHandle.value) {
        case 'nw':
          boxes.value[i] = { start: [x, y], end: [bx2, by2] }
          break
        case 'ne':
          boxes.value[i] = { start: [bx1, y], end: [x, by2] }
          break
        case 'sw':
          boxes.value[i] = { start: [x, by1], end: [bx2, y] }
          break
        case 'se':
          boxes.value[i] = { start: [bx1, by1], end: [x, y] }
          break
      }

      clearCanvas()
      drawBoxes()
      if (i === 0) {
        form.crop_range = boxes.value[0]
      } else {
        form.box_range = boxes.value[1]
      }
    } else if (draggingBoxIndex.value !== null) {
      const i = draggingBoxIndex.value
      const dx = x - dragStart.value.x
      const dy = y - dragStart.value.y
      const box = boxes.value[i]

      boxes.value[i] = {
        start: [box.start[0] + dx, box.start[1] + dy],
        end: [box.end[0] + dx, box.end[1] + dy]
      }

      clearCanvas()
      drawBoxes()
      if (i === 0) {
        form.crop_range = boxes.value[0]
      } else {
        form.box_range = boxes.value[1]
      }
    }
  }
  function onMouseUp() {
    draggingBoxIndex.value = null
    resizingBoxIndex.value = null
    resizeHandle.value = null
  }




  function generate_canvas() {
    const el = document.getElementById('crop-canvas') as HTMLCanvasElement
    if (el) {
      canvas.value = el
      ctx.value = canvas.value.getContext('2d')
      // 设置 canvas 尺寸与视频一致
      const video = document.getElementById('video-player') as HTMLVideoElement
      if (video) {
        if (video.videoWidth > 0 && video.videoHeight > 0) {
          console.log('video size =', video.videoWidth, video.videoHeight)
          canvas.value.width = video.videoWidth
          canvas.value.height = video.videoHeight
          canvas.value.style.zIndex = '2'
          drawBoxes()
        }
      }

      // 绑定事件监听器
      canvas.value.addEventListener('mousedown', onMouseDown)
      canvas.value.addEventListener('mousemove', onMouseMove)
      canvas.value.addEventListener('mouseup', onMouseUp)
      canvas.value.addEventListener('mouseleave', onMouseUp)
    }
  }

  function startDrawing(e: MouseEvent) {
    if (boxes.value.length >= 2) {
      while (boxes.value.length > 2) {
        boxes.value.pop()
      }
      return
    }
    const rect = canvas.value!.getBoundingClientRect()
    isDrawing.value = true
    startX.value = e.clientX - rect.left
    startY.value = e.clientY - rect.top
    endX.value = startX.value
    endY.value = startY.value
    console.log('开始绘制，起点:', startX.value, startY.value)
  }

  function draw(e: MouseEvent) {
    if (!isDrawing.value) return
    const rect = canvas.value!.getBoundingClientRect()
    endX.value = e.clientX - rect.left
    endY.value = e.clientY - rect.top
    clearCanvas()
    drawBoxes()
    drawRect(startX.value, startY.value, endX.value - startX.value, endY.value - startY.value)
  }

  function endDrawing() {
    if (!isDrawing.value) return
    isDrawing.value = false
    boxes.value.push({
      start: [startX.value, startY.value],
      end: [endX.value, endY.value]
    })
    clearCanvas()
    drawBoxes()
    console.log('结束绘制，终点:', endX.value, endY.value)
    console.log('框', boxes.value.length, '个: ', boxes.value)
    if (boxes.value.length === 1) {
      form.crop_range = boxes.value[0]
    }
    if (boxes.value.length === 2) {
      if (boxes.value[1].start[0] < boxes.value[0].start[0] 
        || boxes.value[1].start[1] < boxes.value[0].start[1]
        || boxes.value[1].end[0] > boxes.value[0].end[0]
        || boxes.value[1].end[1] > boxes.value[0].end[1]) {
        ElMessage({
            message: '扫描框必须在裁剪框内',
            type: 'warning'
        })
        reset()
      } else {
        form.box_range = boxes.value[1]
        form.is_set = true
      }
    }
  }

  function drawRect(x: number, y: number, width: number, height: number) {
    const c = ctx.value!
    c.strokeStyle = 'green'
    c.lineWidth = 2
    c.strokeRect(x, y, width, height)
  }

  function drawBoxes() {
    boxes.value.forEach(box => {
      drawRect(box.start[0], box.start[1], box.end[0] - box.start[0], box.end[1] - box.start[1])
    })
  }

  function clearCanvas() {
    const c = canvas.value!
    ctx.value!.clearRect(0, 0, c.width, c.height)
  }

  function reset() {
    if (boxes.value.length === 0) {
      return
    }
    while (boxes.value.length > 0) {
      boxes.value.pop()
    }
    clearCanvas()
    form.crop_range = {}
    form.box_range = {}
    form.is_set = false
    console.log('重置绘制数据')
    // 移除事件监听器
    const el = document.getElementById('crop-canvas') as HTMLCanvasElement
    if (el) {
      el.removeEventListener('mousedown', onMouseDown)
      el.removeEventListener('mousemove', onMouseMove)
      el.removeEventListener('mouseup', onMouseUp)
      el.removeEventListener('mouseleave', onMouseUp)
    }
  }

  return {
    startDrawing,
    draw,
    endDrawing,
    clearCanvas,
    reset,
    generate_canvas
  }
}